### SEIR_simtest

# Plots a comparison of susceptible and infected populations 

# Uses seir_sim (below)

# User specifies values for 6 parameters:
#   R0 is the basic reproductive number
#   TL is the mean incubation time
#   TI is the mean infectious duration
#   E0 is the initial latent fraction
#   I0 is the initial infectious fraction
#   V is the initial immune fraction

# Populations are relative to total population N=1

# by Glenn Ledder
# written 2020/10/15
# revised 2020/11/08

# direct comments to gledder@unl.edu

### seir_sim_test appears below!

#########################################

seir_sim <- function(beta,eta,gamma,E0,I0,V,target)
	{

	## DATA

	maxdays = 1000

	## FUNCTION FOR THE DIFFERENTIAL EQUATION

    	yprime <- function(yy)
		{
      	S = yy[1]
      	E = yy[2]
		I = yy[3]
      	Sp = -beta*S*I
      	Ep = -Sp-eta*E
		Ip = eta*E-gamma*I
      	yp = c(Sp,Ep,Ip)
		return(yp)
		}

	## INITIALIZATION

	# set up results data structure with Y=[S,E,I] and R

	results = matrix(0,nrow=maxdays+1,ncol=4)	
	S0 = 1-E0-I0-V
	Y = c(S0,E0,I0)
	R = V
	results[1,] = c(Y,R)

	oldI = I0

	## FUNCTION FOR rk4

      # y0 is a column vector of initial conditions at t
      # y is a column vector of values at t+dt

	rk4 <- function(dt,y0)
		{
		k1 = yprime(y0)
        	k2 = yprime(y0+0.5*dt*k1)
        	k3 = yprime(y0+0.5*dt*k2)
        	k4 = yprime(y0+dt*k3)
        	yy = y0+dt*(k1+2*k2+2*k3+k4)/6
		return(yy)
		}

	## COMPUTATION

	for (t in 1:maxdays)
		{
		Y = rk4(1,Y)
		R = 1-sum(Y)
    		results[t+1,] = c(Y,R)
    		if (Y[2]+Y[3]>min(target,oldI))
			{
        		oldI = Y[2]+Y[3]
			} else {
	      	results = results[1:(t+1),]
        		break
			}
		}

	## OUTPUT

	return(results)

	## END

	}

#########################################

### seir_sim_test starts here! ###

## SCENARIO DATA

R0 = 2.5
TL = 5
TI = 10
E0 = 0
I0 = 0.001
V = 0

## COMMON DATA

target = 0.001

## INITIALIZATION

par(mfrow=c(1,1))    # set the plotting area into a 1x1 array

## COMPUTATION

eta = 1/TL
gamma = 1/TI
beta = gamma*R0
results = seir_sim(beta,eta,gamma,E0,I0,V,target)
S = results[,1]
E = results[,2]
I = results[,3]
R = results[,4]
days = length(I)-1
new = c(0,S[1:days]-S[2:length(S)])

## OUTPUT

times = 0:days

plot(times, S, type="l", col="green3", ylim=c(0,1), xlab="days", ylab="population fraction", lwd=2, xaxs="i", yaxs="i")
lines(times, E, col="orange", lwd=2)
lines(times, I, col="red", lwd=2)
lines(times, R, col="blue", lwd=2)
legend(150, 0.6, legend=c("S","E","I","R"),col=c("green3","orange","red","blue"), lty=1, lwd=2)

cbind(times,S,E,I,R,new)

# maximum I
max(I)

# final S
S[days+1]

